<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once '../includes/config.php';

header('Content-Type: application/json');

// Start session and check authentication
session_start();
if (!isset($_SESSION['admin_logged_in'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized - Admin session not found']);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

try {
    // Use global PDO
    global $db1;

    $client_id = isset($_GET['client_id']) ? trim($_GET['client_id']) : '';
    $method = isset($_GET['method']) ? trim($_GET['method']) : '';
    $date_from = isset($_GET['date_from']) ? trim($_GET['date_from']) : '';
    $date_to = isset($_GET['date_to']) ? trim($_GET['date_to']) : '';

    $where = [];
    $params = [];

    if ($client_id !== '') {
        $where[] = 'p.client_id = ?';
        $params[] = $client_id;
    }
    if ($method !== '') {
        $where[] = 'p.method = ?';
        $params[] = $method;
    }
    if ($date_from !== '') {
        $where[] = 'p.payment_date >= ?';
        $params[] = $date_from;
    }
    if ($date_to !== '') {
        $where[] = 'p.payment_date <= ?';
        $params[] = $date_to;
    }

    $whereClause = !empty($where) ? ('WHERE ' . implode(' AND ', $where)) : '';

    $sql = "
        SELECT p.*, c.CompanyName as client_name,
               COUNT(pa.id) as allocation_count
        FROM payments p
        LEFT JOIN tbl_clients c ON p.client_id = c.Id
        LEFT JOIN payment_allocations pa ON p.id = pa.payment_id
        {$whereClause}
        GROUP BY p.id
        ORDER BY p.payment_date DESC
    ";

    $stmt = $db1->prepare($sql);
    $stmt->execute($params);
    $payments = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Load currency rates once for allocation conversion fallback
    $rates = [];
    try {
        $rs = $db1->query("SELECT currency_code, units_per_usd FROM currency_rates");
        if ($rs) {
            while ($r = $rs->fetch(PDO::FETCH_ASSOC)) {
                $rates[$r['currency_code']] = (float)$r['units_per_usd'];
            }
        }
    } catch (Exception $e2) {
        // ignore
    }

    // Compute allocated amount per payment in payment currency
    foreach ($payments as &$p) {
        $allocated = 0.0;
        $allocs = $db1->prepare("SELECT pa.amount, pa.rate_to_payment, i.currency FROM payment_allocations pa LEFT JOIN invoices i ON pa.invoice_id = i.id WHERE pa.payment_id = ?");
        $allocs->execute([$p['id']]);
        $rows = $allocs->fetchAll(PDO::FETCH_ASSOC);
        $payUnits = isset($rates[$p['currency']]) ? (float)$rates[$p['currency']] : null;
        foreach ($rows as $row) {
            $rateToPayment = isset($row['rate_to_payment']) && (float)$row['rate_to_payment'] > 0 ? (float)$row['rate_to_payment'] : null;
            if ($rateToPayment === null && $payUnits && isset($rates[$row['currency']])) {
                // Fallback derive rate invoice->payment via USD units
                $rateToPayment = $payUnits / (float)$rates[$row['currency']];
            }
            if ($rateToPayment) {
                $allocated = round($allocated + round(((float)$row['amount']) * $rateToPayment, 2), 2);
            }
        }
        $p['allocated_amount'] = $allocated;
    }
    unset($p);

    echo json_encode(['success' => true, 'payments' => $payments]);
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode(['error' => $e->getMessage()]);
}
?>


