<?php
session_start();
require_once('../includes/config.php');
require_once('../includes/DataBaseObjects.php');
require_once('../includes/cache_handler.php');

// Detect execution context
$isCli = (php_sapi_name() === 'cli');

// Security check (allow CLI/cron without session, enforce for web requests)
if (!$isCli && !isset($_SESSION['admin_logged_in'])) {
    header('Content-Type: application/json');
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit();
}

// CSRF check - only enforce for user-initiated requests (not cron jobs/CLI)
$isUserRequest = isset($_POST['csrf_token']);
if (!$isCli && $isUserRequest && (!isset($_POST['csrf_token']) || !isset($_SESSION['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token'])) {
    header('Content-Type: application/json');
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Invalid security token']);
    exit();
}

// Always set JSON content type header and no-cache for freshness
header('Content-Type: application/json');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');

try {
    // Check cache directory permissions
    $cachePath = dirname(__FILE__) . '/../cache/';
    $cacheFile = $cachePath . 'dashboard_stats.json';
    
    // Ensure cache directory exists
    if (!file_exists($cachePath)) {
        if (!mkdir($cachePath, 0755, true)) {
            throw new Exception('Failed to create cache directory. Check permissions.');
        }
    }
    
    // Check if directory is writable
    if (!is_writable($cachePath)) {
        throw new Exception('Cache directory is not writable. Directory: ' . $cachePath);
    }
    
    // If cache file exists, check if it's writable
    if (file_exists($cacheFile) && !is_writable($cacheFile)) {
        throw new Exception('Cache file exists but is not writable: ' . $cacheFile);
    }
    
    // Create a new DashboardCache instance with debug info
    $cache = new DashboardCache();
    
    // Enable detailed PDO error mode for debugging
    if (isset($db1) && $db1 instanceof PDO) {
        $db1->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    } else {
        throw new Exception('DB connection not initialized');
    }
    
    // Allow optional debug flag
    $debug = false;
    if (isset($_POST['debug'])) { $debug = (int)$_POST['debug'] === 1; }
    if (isset($_GET['debug'])) { $debug = $debug || (int)$_GET['debug'] === 1; }

    // Update the cache
    $dashboardData = $cache->updateCache($db1, $debug);
    
    if ($dashboardData) {
        $response = [
            'success' => true,
            'message' => 'Cache updated successfully',
            'timestamp' => $dashboardData['timestamp'] ?? time(),
            'summary' => $dashboardData['summary'] ?? null,
        ];
        // Include light insights overview counts for convenience
        if (isset($dashboardData['insights'])) {
            $ins = $dashboardData['insights'];
            $response['insights'] = [
                'top_countries_count' => isset($ins['top_countries']) ? count($ins['top_countries']) : 0,
                'top_nationalities_count' => isset($ins['top_nationalities']) ? count($ins['top_nationalities']) : 0,
                'usage_distribution' => $ins['usage_distribution'] ?? null,
                'renewal_buckets' => $ins['renewal_buckets'] ?? null,
                'recent_activity_count' => isset($ins['recent_activity']) ? count($ins['recent_activity']) : 0,
            ];
        }
        echo json_encode($response);
    } else {
        throw new Exception('Failed to update cache - updateCache returned false');
    }
} catch (PDOException $e) {
    // Catch database specific errors
    error_log("Dashboard Cache Update SQL Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'message' => 'Database error: ' . $e->getMessage()
    ]);
} catch (Exception $e) {
    error_log("Dashboard Cache Update Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'message' => 'Failed to update dashboard data: ' . $e->getMessage()
    ]);
}
