<?php
// Check if user is logged in
if (!isset($_SESSION['admin_logged_in'])) {
    header('Location: ../login.php');
    exit;
}

// Get filter parameters
$client_id = $_GET['client_id'] ?? '';
$method = $_GET['method'] ?? '';
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';

// Build query
$where_conditions = [];
$params = [];

if ($client_id) {
    $where_conditions[] = "p.client_id = ?";
    $params[] = $client_id;
}

if ($method) {
    $where_conditions[] = "p.method = ?";
    $params[] = $method;
}

if ($date_from) {
    $where_conditions[] = "p.payment_date >= ?";
    $params[] = $date_from;
}

if ($date_to) {
    $where_conditions[] = "p.payment_date <= ?";
    $params[] = $date_to;
}

$where_clause = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";

// Get payments using existing $db1 connection
$payments = [];
try {
    $query = "
        SELECT p.*, c.CompanyName as client_name,
               COUNT(pa.id) as allocation_count
        FROM payments p
        LEFT JOIN tbl_clients c ON p.client_id = c.Id
        LEFT JOIN payment_allocations pa ON p.id = pa.payment_id
        {$where_clause}
        GROUP BY p.id
        ORDER BY p.payment_date DESC
    ";
    
    $stmt = $db1->prepare($query);
    $stmt->execute($params);
    $payments = $stmt->fetchAll(PDO::FETCH_ASSOC);
    // Compute allocated in payment currency per row
    // Load rates once
    $rates = [];
    try {
        $rs = $db1->query("SELECT currency_code, units_per_usd FROM currency_rates");
        if ($rs) { while ($r = $rs->fetch(PDO::FETCH_ASSOC)) { $rates[$r['currency_code']] = (float)$r['units_per_usd']; } }
    } catch (Exception $e2) {}
    foreach ($payments as &$p) {
        $allocated = 0.0;
        $allocs = $db1->prepare("SELECT pa.amount, pa.rate_to_payment, i.currency FROM payment_allocations pa LEFT JOIN invoices i ON pa.invoice_id = i.id WHERE pa.payment_id = ?");
        $allocs->execute([$p['id']]);
        $rows = $allocs->fetchAll(PDO::FETCH_ASSOC);
        $payUnits = isset($rates[$p['currency']]) ? (float)$rates[$p['currency']] : null;
        foreach ($rows as $row) {
            $rateToPayment = isset($row['rate_to_payment']) && (float)$row['rate_to_payment'] > 0 ? (float)$row['rate_to_payment'] : null;
            if ($rateToPayment === null && $payUnits && isset($rates[$row['currency']])) {
                $rateToPayment = $payUnits / (float)$rates[$row['currency']];
            }
            if ($rateToPayment) {
                $allocated = round($allocated + round(((float)$row['amount']) * $rateToPayment, 2), 2);
            }
        }
        $p['allocated_amount'] = $allocated;
    }
} catch (Exception $e) {
    // Handle error silently for now
    $payments = [];
}

// Get clients for filter
$clients = [];
try {
    $stmt = $db1->prepare("SELECT id, CompanyName FROM tbl_clients ORDER BY CompanyName");
    $stmt->execute();
    $clients = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    // Handle error silently for now
    $clients = [];
}
?>

<!-- Payments Content -->
<div class="d-flex justify-content-between align-items-center mb-4">
    <h2>Payments</h2>
    <a href="?page=payment_create" class="btn btn-primary">Record Payment</a>
</div>

<!-- Filters -->
<div class="card mb-4">
    <div class="card-body">
        <form method="GET" class="row g-3">
            <input type="hidden" name="page" value="payments">
            <div class="col-md-3">
                <label class="form-label">Client</label>
                <select name="client_id" class="form-select">
                    <option value="">All Clients</option>
                    <?php foreach ($clients as $client): ?>
                        <option value="<?= $client['id'] ?>" <?= $client_id == $client['id'] ? 'selected' : '' ?>>
                            <?= htmlspecialchars($client['CompanyName']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-2">
                <label class="form-label">Method</label>
                <select name="method" class="form-select">
                    <option value="">All Methods</option>
                    <option value="cash" <?= $method == 'cash' ? 'selected' : '' ?>>Cash</option>
                    <option value="bank" <?= $method == 'bank' ? 'selected' : '' ?>>Bank</option>
                    <option value="card" <?= $method == 'card' ? 'selected' : '' ?>>Card</option>
                    <option value="transfer" <?= $method == 'transfer' ? 'selected' : '' ?>>Transfer</option>
                    <option value="other" <?= $method == 'other' ? 'selected' : '' ?>>Other</option>
                </select>
            </div>
            <div class="col-md-2">
                <label class="form-label">From Date</label>
                <input type="date" name="date_from" class="form-control" value="<?= $date_from ?>">
            </div>
            <div class="col-md-2">
                <label class="form-label">To Date</label>
                <input type="date" name="date_to" class="form-control" value="<?= $date_to ?>">
            </div>
            <div class="col-md-3 d-flex align-items-end">
                <button type="submit" class="btn btn-secondary me-2">Filter</button>
                <a href="?page=payments" class="btn btn-outline-secondary">Clear</a>
            </div>
        </form>
    </div>
</div>

<!-- Payments Table -->
<div class="card">
    <div class="card-body">
        <div class="table-responsive">
            <table class="table table-striped">
                <thead>
                    <tr>
                        <th>Payment Date</th>
                        <th>Client</th>
                        <th>Amount</th>
                        <th>Currency</th>
                        <th>Method</th>
                        <th>Reference</th>
                        <th>Allocated</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($payments)): ?>
                        <tr>
                            <td colspan="8" class="text-center">No payments found</td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($payments as $payment): ?>
                            <tr>
                                <td><?= $payment['payment_date'] ?></td>
                                <td><?= htmlspecialchars($payment['client_name']) ?></td>
                                <td><?= number_format($payment['amount'], 2) ?></td>
                                <td><?= htmlspecialchars($payment['currency']) ?></td>
                                <td>
                                    <span class="badge bg-info">
                                        <?= ucfirst($payment['method']) ?>
                                    </span>
                                </td>
                                <td><?= htmlspecialchars($payment['reference'] ?: '-') ?></td>
                                <td>
                                    <?= number_format($payment['allocated_amount'] ?? 0, 2) ?>
                                    /
                                    <?= number_format($payment['amount'], 2) ?>
                                </td>
                                <td>
                                    <a href="?page=payment_view&id=<?= $payment['id'] ?>" class="btn btn-sm btn-info">View</a>
                                    <?php if (($payment['allocated_amount'] ?? 0) < $payment['amount']): ?>
                                        <a href="?page=payment_allocate&id=<?= $payment['id'] ?>" class="btn btn-sm btn-warning">Allocate</a>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>
